package kodename.kodeview.launcher.controller;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Properties;
import java.util.ResourceBundle;

import javafx.animation.FadeTransition;
import javafx.application.HostServices;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.concurrent.Task;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.fxml.FXML;
import javafx.geometry.Rectangle2D;
import javafx.scene.Scene;
import javafx.scene.control.Label;
import javafx.scene.layout.AnchorPane;
import javafx.stage.Screen;
import javafx.stage.Stage;
import javafx.stage.StageStyle;
import javafx.util.Duration;
import kodename.config.stubs.ClientVersionDT;
import kodename.config.stubs.ClientVersionResponseDT;
import kodename.config.stubs.ConfigErrorCode;
import kodename.config.stubs.ContextRootDT;
import kodename.config.stubs.JarDependencyDT;
import kodename.kodefile.desktop.Main;
import kodename.kodeview.launcher.Constants;
import kodename.kodeview.launcher.Updater;
import kodename.kodeview.launcher.utils.JarLoader;
import kodename.kodeview.launcher.utils.MD5;
import kodename.kodeview.launcher.utils.OSUtils;
import kodename.kodeview.launcher.viewer.FXMLUrls;
import kodename.kodeview.launcher.webservice.ConfigService;
import kodename.kodeview.launcher.webservice.DesktopLauncherErrorCode;
import kodename.kodeview.launcher.webservice.DesktopLauncherException;
import kodename.kodeview.settings.ProxyConfig;
import kodename.kodeview.settings.Settings;
import kodename.kodeview.settings.SettingsException;
import kodename.kodeview.settings.controller.SettingsController;




public class StartupController {

	@FXML
	private ResourceBundle resources;

	@FXML
	private URL location;

	@FXML
	private AnchorPane anchorPane;

	@FXML
	private Label msgLabel;

	private Stage initialStage;

	private static final int SPLASH_WIDTH = 600;
	private static final int SPLASH_HEIGHT = 318;
	private static String environment;

	public static int APPLICATION_ID;
	private static int MAJOR_VERSION;
	private static int MINOR_VERSION;
	private static int BUILD_VERSION;
	public static String HOST_NAME;
	public static int CONFIG_PORT;
	private static HostServices hostServices;
	private ClientVersionDT clientVersionDt;
	private File proxySettingsFile;
	
	
	@FXML
	private void initialize() {
		assert anchorPane != null : "fx:id=\"anchorPane\" was not injected: check your FXML file 'Splash.fxml'.";
		assert msgLabel != null : "fx:id=\"msgLabel\" was not injected: check your FXML file 'Splash.fxml'.";

		msgLabel.setText("Loading application Settings, please wait..");        

	}

	public void loadSplash(Stage stage, String environment, HostServices hostServices){
		String kodenameCss;
		Scene splashScene;
		URL kodenameCssUrl;
		StartupController.environment = environment;
		StartupController.hostServices = hostServices;
		
		kodenameCssUrl = FXMLUrls.class.getResource(FXMLUrls.MAIN_BSS);
		if(kodenameCssUrl ==null){
			kodenameCssUrl = FXMLUrls.class.getResource(FXMLUrls.MAIN_CSS);
		}

		if(kodenameCssUrl != null ){
			kodenameCss = kodenameCssUrl.toExternalForm();
		}else{
			kodenameCss = null;
		}

		this.initialStage = stage;

		splashScene = new Scene(anchorPane);
		if(kodenameCss != null){
			splashScene.getStylesheets().add(kodenameCss);
		}

		initialStage.initStyle(StageStyle.UNDECORATED);
		initialStage.getIcons().add(Constants.ICON_IMAGE_16);
		initialStage.getIcons().add(Constants.ICON_IMAGE_32);
		initialStage.setTitle("KodeFile");
		
		
		
		final Rectangle2D bounds = Screen.getPrimary().getBounds();
		initialStage.setScene(splashScene);
		// center the splash
		initialStage.setX(bounds.getMinX() + bounds.getWidth() / 2 - SPLASH_WIDTH / 2);
		initialStage.setY(bounds.getMinY() + bounds.getHeight() / 2 - SPLASH_HEIGHT / 2);
		initialStage.show();

		startApplication();
		
	}

	private void startApplication(){
		
		try{
			Task<Void> startupTask = initStartApplicationTask();

			startupTask.exceptionProperty().addListener(new ChangeListener<Throwable>(){

				@Override
				public void changed(ObservableValue<? extends Throwable> observable, Throwable oldValue, Throwable newValue) {
					newValue.printStackTrace();
					StartupErrorMessageController controller;
					controller = (StartupErrorMessageController)FXMLControllerLoader.loadController(FXMLUrls.STARTUP_ERROR_MESSAGE);
					
					if(newValue instanceof DesktopLauncherException){
						DesktopLauncherException e = (DesktopLauncherException)newValue;
						if(e.getErrorCode() == DesktopLauncherErrorCode.PROXY_ERROR){
							controller.showDialog(initialStage, e.getErrorTitle(), e.getErrorMessage());
							showProxySettings();
						}else{
							controller.showDialog(initialStage, e.getErrorTitle(), e.getErrorMessage());
							System.exit(-1);
						}
					}else{
						controller.showDialog(
								initialStage,
								"Startup Failure", 
								"The application failed to launch due to the following error:\n"+newValue.getMessage());
					
						System.exit(-1);
					}
					
				}

			});
			msgLabel.textProperty().bind(startupTask.messageProperty());
			new Thread(startupTask).start();

		}catch(Exception e){
			e.printStackTrace();
		}
	}
	

	private void showProxySettings(){
		SettingsController controller;
		controller = Settings.initSettingsController();
		// show dialog is a blocking call.
		controller.showDialog(initialStage, Main.hostServices, proxySettingsFile);
		// check for any proxy errors
		startApplication();	
	}

	private void loadApplication() throws IOException{
		Main mainApp;
		Properties properties;
		properties = new Properties();
		for(ContextRootDT dt: clientVersionDt.getContextRootDts()){
			if(dt.getPortName().compareTo("account_port")==0){
				properties.put("account_port", dt.getPortNumber());
				properties.put("account_context_root", dt.getContextRoot());
				
			}else if(dt.getPortName().compareTo("kodefile_port")==0){
				properties.put("kodefile_port", dt.getPortNumber());
				properties.put("kodefile_context_root", dt.getContextRoot());
				
			}else{
				throw new RuntimeException("Unknown Application Port: "+ dt.getPortName());
			}
			
			
		}
		
		properties.put("application_id", Integer.toString(APPLICATION_ID));
		properties.put("hostname", HOST_NAME);
		properties.put("version", clientVersionDt.getVersionName());
		properties.put("environment", environment);
		mainApp = new Main(properties, hostServices);
		mainApp.loadMainApplication();		
		unloadSplash();
	}


	private Task<Void> initStartApplicationTask(){
		Task<Void> task;
		task = new Task<Void>(){

			@Override
			protected Void call() throws Exception {
				File applicationDirectory;
				File kodeFileDirectory;
				ClientVersionResponseDT responseDt;
				Updater updater;
				ConfigService configService;
				String configUrl;
				ConfigErrorCode errorCode;
				updateMessage("Loading launcher properties");
				kodeFileDirectory = initializeKodeFileDirectory();
				initializeSettingsFile(kodeFileDirectory);
				loadProxySettings();
				loadLauncherProperties();
				updateMessage("Initializing Application Directory");				
				applicationDirectory = initializeApplicationDirectory(kodeFileDirectory);

				updateMessage("Connecting to server..");
				if(CONFIG_PORT == 443){
					configUrl = HOST_NAME + "/configwebv100/ConfigWSEPService?wsdl";
				}else{
					configUrl =HOST_NAME + ":"+ CONFIG_PORT + "/configwebv100/ConfigWSEPService?wsdl";
				}
				configService = new ConfigService(new URL(configUrl));
				updateMessage("getting configuration information....");
				responseDt = configService.getLatestVersion(APPLICATION_ID, OSUtils.getOS());
				
				if((errorCode = responseDt.getErrorCode()) != null){
					switch(errorCode){
					case CONFIG_NOT_FOUND:
						throw new DesktopLauncherException(
								DesktopLauncherErrorCode.CONFIG_NOT_FOUND,
								"Configuration Not Found",
								"Configuration information for client not available at this time. Please retry later.");
						
					case CLIENT_NOT_SUPPORTED:
						throw new DesktopLauncherException(
								DesktopLauncherErrorCode.CLIENT_NOT_SUPPORTED,
								"Client Not Supported",
								"The version of the client you are using is no longer supported. Please download and "
								+ "install the latest version from http://www.1sqtechnologies.com");						
					}

				}
				clientVersionDt = responseDt.getVersionDt();
				updateMessage("verifying installed components..");
							
				if(isReInstallationRequired(clientVersionDt)){
					throw new RuntimeException("Re-Install Required");
				}
				updateMessage("Checking if update is required");
				if(isJarUpdateAvailable(applicationDirectory, clientVersionDt)){
					updateMessage("Installing updates.....");
					updater = new Updater(applicationDirectory, responseDt.getVersionDt().getJarDt());
					updater.update();
				}else if(!verifyApp(applicationDirectory, clientVersionDt)){
					updateMessage("Unable to verify Installed components, re-installing .....");
					updater = new Updater(applicationDirectory, responseDt.getVersionDt().getJarDt());
					updater.update();
				}else{
					updateMessage("application components up to date");
				}
				
				updateMessage("Loading Application....");
				
				loadJars(applicationDirectory);
				return null;
			}
			
			
			
		

			@Override
			protected void succeeded(){
				try{
					loadApplication();
				}catch(IOException e){
					e.printStackTrace();
				}
			}

			private List<String> loadJars(File applicationDirectory) throws Exception{
				List<String> loadedJars;
				loadedJars = JarLoader.loadJars(applicationDirectory);			
				return loadedJars;
			}
			
			private boolean isReInstallationRequired(ClientVersionDT versionDt){
				return false;
			}
			
			private boolean isJarUpdateAvailable(File applicationDirectory, ClientVersionDT dt){
				try{
					if(!loadProperties(applicationDirectory)){
						return true;
					}
					if(dt.getMajorNumber() > MAJOR_VERSION ||
							(dt.getMajorNumber() >= MAJOR_VERSION && dt.getMinorNumber() > MINOR_VERSION) ||
							(dt.getMajorNumber() >= MAJOR_VERSION && dt.getMinorNumber() >= MINOR_VERSION && dt.getBuildNumber() > BUILD_VERSION)){
					
						return true;
					}
					
				}catch(IOException e){
					return true;
				}
				return false;
			}
			
			
			/**
			 * Check to make sure all dependencies are available
			 * @param applicationDirectory
			 * @param dt
			 * @return
			 */
			private boolean verifyApp(File applicationDirectory, ClientVersionDT dt) throws Exception{
				File lib;
				File jarFile;
				String jarFileMD5;
				
				if(applicationDirectory == null || !applicationDirectory.exists() || applicationDirectory.isFile()){
					return false;
				}
				
				// verify the main application jar
				if(! (jarFile = new File(applicationDirectory, dt.getJarDt().getJarFileName())).exists()){
					return false;
				}else{
					jarFileMD5 = MD5.getDigest(jarFile);
					if(jarFileMD5.compareTo(dt.getJarDt().getJarFileMD5())!=0){
						return false;
					}
				}
				
				// if library folder does not exist, fail 
				if(!(lib = new File(applicationDirectory, "lib")).exists()){
					return false;
				}

				for(JarDependencyDT dependencyDt: dt.getJarDt().getDependencyDts()){
					if(! (jarFile=new File(lib, dependencyDt.getJarName())).exists()){
						return false;
					}else{
						jarFileMD5 = MD5.getDigest(jarFile);
						if(jarFileMD5.compareTo(dependencyDt.getJarMD5())!=0){
							return false;
						}
					}
				}			
				return true;
			}
						
		};
		return task;
	}
	
	private File initializeKodeFileDirectory(){
		File userHomeDirectory=null;
		File kodefileDirectory=null;

		userHomeDirectory = new File(System.getProperty("user.home"));
		if(userHomeDirectory.exists()){
			kodefileDirectory = new File(userHomeDirectory, ".kodefile");
			if(!kodefileDirectory.exists()){
				if(!kodefileDirectory.mkdir()){
					throw new RuntimeException("Unable to create kodefile directory at "+ kodefileDirectory.getAbsolutePath());
				}
			}
		}
		return kodefileDirectory;
		
	}

	private File initializeApplicationDirectory(File kodefileDirectory) throws IOException{
		File applicationDirectory=null,kodenameSettingsFile=null;

		kodenameSettingsFile = new File(kodefileDirectory, "settings.properties");
		applicationDirectory = new File(kodefileDirectory, "app");
		if(!applicationDirectory.exists()){
			if(!applicationDirectory.mkdir()){
				throw new RuntimeException("Unable to create application directory at "+ applicationDirectory.getAbsolutePath());
			}
		}
		if(!kodenameSettingsFile.exists()){
			kodenameSettingsFile = new File(kodefileDirectory, "settings.properties");
		}


		return applicationDirectory;				
	}
	
	
	private File initializeSettingsFile(File kodefileDirectory) throws IOException{
		//File kodenameSettingsFile=null;
		
		proxySettingsFile = new File(kodefileDirectory, "settings.properties");
		if(!proxySettingsFile.exists()){
			proxySettingsFile = new File(kodefileDirectory, "settings.properties");
		}


		return proxySettingsFile;				
	}

	private void loadProxySettings() throws SettingsException{
		new ProxyConfig(proxySettingsFile);
	}
	
	private void loadLauncherProperties() throws IOException{
		Properties properties;
		properties = new Properties();
		properties.load(Updater.class.getResourceAsStream("kodename.properties."+ environment));
		APPLICATION_ID = Integer.parseInt(properties.getProperty("kodename.application_id"));
		CONFIG_PORT  = Integer.parseInt(properties.getProperty("kodename.config_port"));
		HOST_NAME = properties.getProperty("kodename.host");

	}


	private boolean loadProperties(File applicationDirectory) throws IOException{
		Properties properties;
		String propertiesFileName;
		boolean appJarFileFound=false;
		propertiesFileName = "desktop.properties";
		if(!applicationDirectory.exists()){
			return false;
		}
		if(applicationDirectory.isDirectory()){
			// locate the kodefile jar
			for(File f: applicationDirectory.listFiles()){
				
				if(f.isFile() && f.getName().startsWith("desktop-viewer")){
					appJarFileFound = true;
					properties = JarLoader.getProperties(f, propertiesFileName);
					if(properties !=null){
						APPLICATION_ID 			= Integer.parseInt(properties.getProperty("application_id"));
						MAJOR_VERSION 			= Integer.parseInt(properties.getProperty("major_version"));
						MINOR_VERSION 			= Integer.parseInt(properties.getProperty("minor_version"));
						BUILD_VERSION 			= Integer.parseInt(properties.getProperty("build_version"));
					}else{
						throw new RuntimeException("Failed to Load Properties");
					}
					break;
				}
			}
		}else{
			throw new RuntimeException("Unexpected Error Starting application..");
		}
		return appJarFileFound;
	}
	
	 
	private void unloadSplash(){
		FadeTransition fadeSplash = new FadeTransition(Duration.seconds(1.2), anchorPane);
		fadeSplash.setFromValue(1.0);
		fadeSplash.setToValue(0.0);

		fadeSplash.setOnFinished(new EventHandler<ActionEvent>(){
			@Override
			public void handle(ActionEvent arg0) {
				initialStage.hide();
			}

		});
		fadeSplash.play();
	}

	

	/*
	private  void showShutdownDialog(Stage stage, String title, String message, boolean threaded, boolean showCautionLabel){	
		MessageDialogController controller;
		controller = (MessageDialogController)FXMLControllerLoader.loadController(FXMLUrls.MESSAGE_DIALOG);			
		controller.showDialog(stage, title, message, "Quit", showCautionLabel);			
		return;
	}


	private void showStartupErrorMessage(Stage stage, String title, String message){	
		StartupErrorMessageController controller;
		controller = (StartupErrorMessageController)FXMLControllerLoader.loadController(FXMLUrls.STARTUP_ERROR_MESSAGE);			
		controller.showDialog(stage, title, message);
	}
	 */

	/**
	 * returns true iff there are new proxy settings saved.
	 * Retry re-rerunning the config
	 * @param stage
	 * @param title
	 * @param message
	 * @return
	 */

	/*
	private boolean showProxySettingErrorMessage(Stage stage, String title, String message){	
		ConfirmationDialogController controller;
		controller = (ConfirmationDialogController)FXMLControllerLoader.loadController(FXMLUrls.CONFIRMATION_DIALOG);			
		if(!controller.showDialog(stage, title, "Edit Settings", "Quit", message)){
			if(!showProxySettings(stage)){
				return showProxySettingErrorMessage(stage, title, message);
			}
			return true;
		}
		return false;
	}
	 */

	/**
	 * displays a window to edit the proxy settings and returns true if 
	 * settings have been updated. Else, returns false.
	 * @param stage
	 * @return
	 */

	/*
	private boolean showProxySettings(Stage stage){	
		final DialogWindow dialogWindow;

		SettingsController controller;
		dialogWindow = new DialogWindow(stage, "Settings");

		controller =  ((SettingsController)FXMLControllerLoader.loadController(FXMLUrls.SETTINGS));
		controller.cancelledProperty().addListener(new ChangeListener<Boolean>(){
			@Override
			public void changed(ObservableValue<? extends Boolean> observable, Boolean oldValue, Boolean newValue) {
				dialogWindow.hide();
			}
		});
		dialogWindow.createDialog(controller.getPane());
		dialogWindow.show();
		return false;
	}


	 */

	
	
	
}
