package kodename.kodeview.launcher;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.security.MessageDigest;
import java.util.zip.GZIPInputStream;

import org.apache.commons.compress.archivers.ArchiveException;
import org.apache.commons.compress.archivers.ArchiveStreamFactory;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.utils.IOUtils;

import kodename.config.stubs.JarDT;
import kodename.kodeview.launcher.utils.MD5;


public class Updater {

	private File installationDirectory;
	private JarDT jarDt;

	public Updater(File installationDirectory, JarDT jarDt){
		this.installationDirectory = installationDirectory;
		this.jarDt = jarDt;
	}


	private File downloadUpdate() throws Exception{
		URLConnection connection;
		InputStream inputStream;
		File tmpFolder;
		File installer;
		File downloadDirectory;
		FileOutputStream tmpOutputStream;
		URL downloadURL;
		int bufLen = 1024 * 1024;
		int inputLength;
		byte[] plainTextBuffer;
		byte[] plainTextMD5Buffer;
		String md5;

		MessageDigest plainTextHash;
		inputStream = null;
		tmpOutputStream = null;

		try {
			downloadURL = new URL(jarDt.getUpdateUrl() + jarDt.getUpdateFileName());
			connection = downloadURL.openConnection();	

			// get input stream for file being downloaded
			inputStream = connection.getInputStream();
			// create a tmp file to locate the tmp folder
			tmpFolder = new File(System.getProperty("java.io.tmpdir"));
			// if there is already a file downloaded with this name, version it.
			downloadDirectory = new File(
					tmpFolder,
					"kodeViewUpdater"+ 
							System.currentTimeMillis()+ 
							Double.toString(Math.random()));
			//System.out.println("Download Directory: "+ downloadDirectory.getAbsolutePath());
			if(!downloadDirectory.mkdir()){
				throw new RuntimeException("Unable to save downloaded file in "+ downloadDirectory.getAbsolutePath());
			}

			installer = new File(downloadDirectory, jarDt.getUpdateFileName());		
			//System.out.println("Installer FileName: "+ installer.getAbsolutePath());
			tmpOutputStream = new FileOutputStream(installer);
			plainTextHash = MessageDigest.getInstance(Constants.MESSAGE_DIGEST);
			// create input buffer
			plainTextBuffer = new byte[bufLen];
			while ((inputLength = inputStream.read(plainTextBuffer)) != -1) {
				tmpOutputStream.write(plainTextBuffer,0, inputLength);
				// update the digest for the plain text
				plainTextHash.update(plainTextBuffer, 0, inputLength);
			}
			// close the input stream and tmp output stream;
			inputStream.close();
			tmpOutputStream.close();

			inputStream = null;
			tmpOutputStream = null;
			// compute the md5 for the plain text. NEVER disclose this plain text md5 on the encypted file.
			plainTextMD5Buffer = plainTextHash.digest();
		} catch(FileNotFoundException e){
			e.printStackTrace();
			throw new RuntimeException("Download Error: Unable to update client because update installer not found on server.");
		}catch(IOException e){
			e.printStackTrace();
			String errMsg = e.getMessage();
			if(errMsg.indexOf("There is not enough space on the disk") != -1 ||
					errMsg.indexOf("Not enough space") !=-1 ||
					errMsg.indexOf("No space left on device") != -1){		
				throw new RuntimeException("Disk Full");
			}else{
				throw new RuntimeException("Error reading/writing file: "+ e.getMessage());
			}
		}catch(Exception e){
			e.printStackTrace();
			throw new RuntimeException("Unexpected Error: " + e.getMessage());
		}finally {
			if (inputStream != null) {
				try {
					inputStream.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
			if (tmpOutputStream != null) {
				try{
					tmpOutputStream.close();
				}catch(IOException e){
					e.printStackTrace();
				}
			}
		}
		md5 = MD5.byteArrayToHexString(plainTextMD5Buffer); 
		if(md5.compareToIgnoreCase(jarDt.getUpdateFileMD5())!=0){
			throw new RuntimeException("MD5 Does NOT MATCH.");
		}
		return installer;
	}


	public void update() throws Exception{
		File updateJarPkg;
		// download the file a
		updateJarPkg = downloadUpdate();
		// extract the files
		installUpdate(updateJarPkg);
	}

	private boolean installUpdate(File installer){
		File tarball;
		String appDirectoryPath;
		File appDirectoryBackUp;
		if(installer == null){
			throw new RuntimeException("Installer is NULL");
		}
		appDirectoryPath = installationDirectory.getAbsolutePath();
		appDirectoryBackUp = new File(installationDirectory.getParentFile(), installationDirectory.getName()+".bak");
		installationDirectory.renameTo(appDirectoryBackUp);		
		installationDirectory = new File(appDirectoryPath);
		if(!installationDirectory.mkdir()){
			throw new RuntimeException("Unable to create installation directory at "+ 
					installationDirectory.getAbsolutePath());
		}
		try{
			// gunzip the tarball
			tarball = unGZip(installer);
			// untar 
			unTar(tarball);
			// remove the backup
			deleteFile(appDirectoryBackUp);
			return true;
		}catch(FileNotFoundException e1){
			e1.printStackTrace();
		}catch(IOException e2){
			e2.printStackTrace();
		}catch(ArchiveException e3){
			e3.printStackTrace();
		}
		return false;
	}


	private void deleteFile(File element) {
		if (element.isDirectory()) {
			for (File sub : element.listFiles()) {
				deleteFile(sub);
			}
		}
		element.delete();
	}


	private File unGZip(File gzippedUpdater) throws IOException{
		File tarFile;
		String tarFileName;

		tarFileName = gzippedUpdater.getName();
		tarFileName = tarFileName.substring(0, tarFileName.lastIndexOf(".gz"));
		tarFile = new File(gzippedUpdater.getParentFile(), tarFileName);

		final GZIPInputStream in = new GZIPInputStream(new FileInputStream(gzippedUpdater));
		final FileOutputStream out = new FileOutputStream(tarFile);

		IOUtils.copy(in, out);

		in.close();
		out.close();

		return tarFile;
	}


	private void unTar(final File tarFile) throws FileNotFoundException, IOException, ArchiveException {
		final InputStream is = new FileInputStream(tarFile); 
		final TarArchiveInputStream debInputStream = (TarArchiveInputStream) new ArchiveStreamFactory().createArchiveInputStream("tar", is);
		TarArchiveEntry entry = null; 

		while ((entry = (TarArchiveEntry)debInputStream.getNextEntry()) != null) {
			final File outputFile = new File(installationDirectory, entry.getName());
			if (entry.isDirectory()) {
				if (!outputFile.exists()) {
					if (!outputFile.mkdirs()) {
						throw new IllegalStateException(String.format("Couldn't create directory %s.", outputFile.getAbsolutePath()));
					}
				}
			} else {
				final OutputStream outputFileStream = new FileOutputStream(outputFile); 
				IOUtils.copy(debInputStream, outputFileStream);
				outputFileStream.close();
			}

		}
		debInputStream.close(); 
	}


}
