package kodename.kodeview.settings;

import java.io.File;
import java.io.IOException;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URL;
import java.util.List;
import java.util.Properties;

public class ProxyConfig {
	private final File settingsFile;
	private final Properties proxyProps; 
	public static enum PROXY_CONFIG {MANUAL, SYSTEM, NO_PROXY};


	public ProxyConfig(File settingFile){
		this.settingsFile = settingFile;
		proxyProps = new Properties();

		initProxySettings();
	}

	private void initProxySettings(){

		Properties proxyProps;
		String proxy_settings_type;
		try{
			if(settingsFile.exists()){
				// load the properties file
				proxyProps = PropertiesManager.loadParams(settingsFile);
				proxy_settings_type = proxyProps.getProperty("proxy_settings_type");
				switch(proxy_settings_type){
				case "no_proxy":
					configProxySettings(PROXY_CONFIG.NO_PROXY);
					break;
				case "system":
					configProxySettings(PROXY_CONFIG.SYSTEM);
					break;
				case "manual":
					configProxySettings(
							proxyProps.getProperty("http_host"),
							proxyProps.getProperty("http_port"),
							proxyProps.getProperty("https_host"),
							proxyProps.getProperty("https_port")						
							);
					break;
				default:

				}

			}else{
				noProxySettings();
			}
		}catch(SettingsException e){
			noProxySettings();
		}
	}

	private void noProxySettings(){
		proxyProps.clear();
		proxyProps.put("proxy_settings_type", "no_proxy");

		PropertiesManager.saveParamChanges(proxyProps, settingsFile);
		new CustomProxySelector().setNoProxy();
	}

	private void manuallyConfigureProxySettings(
			String httpProxyHost, 
			String httpProxyPort, 
			String httpsProxyHost, 
			String httpsProxyPort)throws SettingsException{

		int httpsPort;
		int httpPort;
		
		httpsPort = getPortNumber(httpsProxyPort);
		httpPort = getPortNumber(httpProxyPort);
		testConnection(httpsProxyHost, httpsPort);
		testConnection(httpsProxyHost, httpsPort);
		
		new CustomProxySelector().setManualProxy(httpsProxyHost, httpsPort, 
				httpProxyHost, httpPort);
		proxyProps.clear();
		proxyProps.put("proxy_settings_type", "manual");
		proxyProps.put("http_host", httpProxyHost);
		proxyProps.put("http_port", httpProxyPort);
		proxyProps.put("https_host", httpsProxyHost);
		proxyProps.put("https_port", httpsProxyPort);	

		PropertiesManager.saveParamChanges(proxyProps, settingsFile);


	}

	private int getPortNumber(String portNumber) throws SettingsException{
		try{
			return Integer.parseInt(portNumber);
		}catch(NumberFormatException e){
			throw new SettingsException(
					SettingsErrorCode.INVALID_PORT, 
					"Invalid Proxy Port", 
					"Unable to connect to proxy server because the port number " + portNumber+ " is invalid. Please verify the proxy server port number in your settings.");
		}
	}
	
	private void testConnection(String hostName, int portNumber) throws SettingsException{
		InetSocketAddress socketAddress;
		try{
			socketAddress = new InetSocketAddress(hostName, portNumber);
			if(socketAddress.isUnresolved()){
				throw new SettingsException(
						SettingsErrorCode.UNKNOWN_HOST, 
						"Unknown Proxy Host", 
						"Unable to connect to proxy host server at address " + hostName + ". Please verify the proxy server address in your settings. Also please verify that the proxy server is running.");
			}
			Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(hostName, portNumber));
			URL url = new URL("https://www.1sqtechnologies.com");
			
			HttpURLConnection uc = (HttpURLConnection)url.openConnection(proxy);
			uc.connect();
		}catch(IllegalArgumentException e){
			throw new SettingsException(
					SettingsErrorCode.INVALID_PORT, 
					"Invalid Proxy Port", 
					"Unable to connect to proxy server because the port number " + portNumber+ " is invalid. Please verify the proxy server port number in your settings.");
		}catch(MalformedURLException e){
			
		}catch(ConnectException e){
			if(e.getMessage().toLowerCase().contains("connection refused")){
				throw new SettingsException(
						SettingsErrorCode.CONNECTION_REFUSED, 
						"Connection Refused", 
						"Unable to connect to proxy server because the the server refused connection at port number" + portNumber+ ". Please verify the proxy server port number in your settings.");
				
			}else{
				throw new SettingsException(
						SettingsErrorCode.UNEXPECTED_ERROR, 
						"Proxy Configuration Error", 
						"Unable to configure proxy settings due to an unexpected error:\n"+
						e.getMessage());
				
			}
			
		}catch(IOException e){
			if(e.getMessage().contains("403 Forbidden") || e.getMessage().contains("Unable to tunnel through proxy")){
				throw new SettingsException(
						SettingsErrorCode.CONNECTION_BLOCKED_BY_PROXY, 
						"Connection Blocked by Proxy",
						"Unable to connect to the internet due to a proxy server that is refusing connections.\n"+
						"Please Check the proxy settings to make sure that they are correct.\n"+
						"You may also contact your network administrator to make sure the proxy server is working.");
								
			}else{
				throw new SettingsException(
						SettingsErrorCode.UNEXPECTED_ERROR, 
						"Proxy Configuration Error", 
						"Unable to configure proxy settings due to an unexpected error:\n"+
						e.getMessage());
			}
			
		}
	}



	private void useSystemProxySettings() throws SettingsException{		

		System.setProperty("java.net.useSystemProxies", "true");
		new CustomProxySelector().setDefaultSystemProxy();
		testSystemProxySettings();
		proxyProps.clear();
		proxyProps.put("proxy_settings_type", "system");
		PropertiesManager.saveParamChanges(proxyProps, settingsFile);		

	}

	public Properties  getProxySettings(){
		return proxyProps;
	}

	private void testSystemProxySettings() throws SettingsException{
		for(Proxy proxy: getProxy("http://www.1sqtechnologies.com")){
			if(proxy != null){
				System.out.println("Proxy Type: "+ proxy.type());
				InetSocketAddress proxyAddress = (InetSocketAddress) proxy.address();
				if(proxyAddress != null){
					testConnection(proxyAddress.getHostName(), proxyAddress.getPort());
				}
			}else{
				System.out.println("Proxy is null");
			}
		}
	}

	private List<Proxy> getProxy(String uri) {
		try {
			return ProxySelector.getDefault().select(new URI(uri));
		} catch (Exception e) {
			e.printStackTrace();
		}

		return null;
	}


	public void configProxySettings(PROXY_CONFIG settings) throws SettingsException{
		switch(settings){
		case SYSTEM:
			useSystemProxySettings();
			break;
		case NO_PROXY:
			noProxySettings();
			break;
		default:

		}

	}

	public void configProxySettings(String httpHost, String httpPort, String httpsHost, String httpsPort) throws SettingsException{
		System.setProperty("java.net.useSystemProxies", "false");
		manuallyConfigureProxySettings(httpHost, httpPort, httpsHost, httpsPort);		
	}

}
